/********************************************************************
 * (C) Copyright 1998 by Hewlett-Packard GmbH. All rights reserved. *
 ********************************************************************/

/***********************************************************************
* File name     : master.c
************************************************************************
* Created by    : Joerg Weedermann, BID R&D, 05.06.96
* Last modified : Tilmann Wendel, adding fcts for the dynamic capabilities
************************************************************************
* File contents :
* Master control functions
************************************************************************/

#if defined(_MSC_VER)
# pragma data_seg("master_c","FAR_DATA")
#elif defined(__BORLANDC__)
# pragma option -dc
#endif /* _MSC_VER etc. */


#ifdef BEST_FIRMWARE

#include <capitype.h>


#else /* BEST_FIRMWARE */

#include <typedefs.h>

/* TODO: remove as soon as puprpget has found a good place */
#include <mini_api.h>

#endif  /* BEST_FIRMWARE */

#include <decode.h>
#include <dynamic.h>
#include <errcapi.h>
#include <iocommon.h>
#include <ladata.h>
#include <master.h>
#include <regconst.h>
#include <regx20.h>
#include <regx22.h>
#include <regx23.h>
#include <regx50.h>
#include <regx52.h>
#include <regx53.h>
#include <session.h>


/* some defines for the master attribute functions */
#define M_WAITS_BITS         0x0000001F
#define M_DOLOOP_BIT         0x00000080
#define M_LAST_BIT           0x00000040
#define M_RELREQ_BIT         0x00000020
#define M_DPERR_BIT          0x00000100
#define M_APERR_BIT          0x00000200
#define M_DWRPAR_BIT         0x00000400
#define M_AWRPAR_BIT         0x00000800
#define M_DTOGGLE_BIT        0x00002000
#define M_ATOGGLE_BIT        0x00001000
#define M_LOCK_BITS          0x0000C000
#define M_LOCK_LOCK_BIT      0x00004000
#define M_LOCK_HIDELOCK_BIT  0x0000C000
#define M_LOCK_UNLOCK_BIT    0x00008000
#define M_DSERR_BIT          0x00010000UL

/* stop compiler/lint warnings */
static const int ABORT = 1;


/* Generic info for range checking */
static b_generic_infotype *GenInfo;


/* --------------------------------------------------------------------------
 * Master Block Property Programming Functions
 * ------------------------------------------------------------------------- */
/* initializes a master block page                                           */
b_errtype EXPORT BestMasterBlockPageInit(
    b_handletype handle,
    b_int32 page_num)
{
#ifndef E2921A_MIN
  B_DECLARE_FUNCNAME ("BestMasterBlockPageInit [mbpginit]");
#endif
  B_TRY_VARS_NO_PROG;
  b_int8 zw8 = (b_int8)page_num;
  
  B_TRY_BEGIN {

#ifndef BEST_FIRMWARE
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_BLOCK, &GenInfo));

    /* Check range of page_num */
    if (page_num == 0 || page_num > GenInfo->num_pages) {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_MBLOCK_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }

    if (Best16BitRegisterFile(handle)) {

#endif  /* BEST_FIRMWARE */
      
      B_TRY(BestBasicCommand(handle, CMD_EXE_MBLK_PAGEINIT,
			     &zw8, IN_EXE_MBLK_PAGEINIT,
			     NULL, NULL));

#ifndef BEST_FIRMWARE
    }

    else {
      B_TRY(BestBasicBlockWrite(handle, MBLK_PROG_PAGE_NUM,
				&zw8, 1UL));
    }

#endif  /* BEST_FIRMWARE */

  }
  
  B_ERRETURN(B_TRY_RET);
}


#ifndef BEST_FIRMWARE

/* --------------------------------------------------------------------------
 * sets the default value of the block properties *
 * does not exist in firmware implementation *
 * ------------------------------------------------------------------------- */

b_errtype EXPORT BestMasterBlockPropDefaultSet(
    b_handletype handle)
{
  B_TRY_VARS_NO_PROG;
  b_param_infotype *ParamInfo;
  b_int32 i;
  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_BLOCK, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      B_TRY(BestParamInfoGet(handle, B_PARAM_MASTER_BLOCK, i, &ParamInfo,
	  (b_int32)B_INDEX_SEARCH));

      /* Set it to default */
      B_TRY(BestMasterBlockPropSet(handle, ParamInfo->proptyp.blockprop,
          ParamInfo->defaultval));
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


/* read the master block properties */
b_errtype EXPORT BestMasterBlockPropGet(
    b_handletype handle,
    b_blkproptype blk_prop,
    b_int32 * pValue)
{
  B_DECLARE_FUNCNAME("BestMasterBlockPropGet [mbprpget]");

  B_TRY_VARS_NO_PROG;
  b_int8 zw8;
  
  /* This command is not available on E2925A */
  if (BestIs2925(handle))
  {
    B_ERRETURN(B_E_NOT_E2925A); 
  }

  B_TRY_BEGIN {

    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    /* Get pointer to i-th (existing) property */
    B_TRY(BestParamProbe(handle, B_PARAM_MASTER_BLOCK, (b_int32)blk_prop));
    B_TRY_FCT_PARAM_NULL_POINTER(pValue);

    if (Best16BitRegisterFile(handle)) {
      /* Get current value */
      B_TRY (BestAbstractPropLongGet(handle, CMD_EXEFHIF_MBLK_PROP_GET,
				     (b_int8)blk_prop, pValue));
    }
    
    else {
      switch (blk_prop) {
       case B_BLK_BUSADDR:
        B_TRY(BestBasicBlockRead(handle, BUS_ADDR_LO, (b_int8ptr) pValue, 4UL));
        break;

       case B_BLK_INTADDR:
        B_TRY(BestBasicBlockRead(handle, INT_ADDR, (b_int8ptr) pValue, 4UL));
        *pValue = *pValue << 2;
        break;

       case B_BLK_BUSCMD:
        B_TRY(BestBasicBlockRead(handle, BUS_CMD, &zw8, 1UL));
        *pValue = (b_int32) zw8;
        break;

       case B_BLK_BYTEN:
        B_TRY(BestBasicBlockRead(handle, BYTEN, &zw8, 1UL));
        *pValue = (b_int32) zw8;
        break;

       case B_BLK_NOFDWORDS:
        B_TRY(BestBasicBlockRead(handle, N_OF_TRANSFERS, (b_int8ptr) pValue, 4UL));
        break;

       case B_BLK_ATTRPAGE:
        B_TRY(BestBasicBlockRead(handle, MATTR_RUN_PAGE_NUM, (b_int8ptr) pValue, 4UL));
        break;

       case B_BLK_COMPOFFS:
        B_TRY(BestBasicBlockRead(handle, COMPARE_OFFSET, (b_int8ptr) pValue, 4UL));
        *pValue = *pValue << 2;
        break;

       case B_BLK_COMPFLAG:
        B_TRY(BestBasicBlockRead(handle, COMPARE_CONTROL, (b_int8ptr) & zw8, 1UL));
        *pValue = (b_int32) zw8;
        break;

       default:
        B_TRY_FCT_PARAM_R(2, ABORT, "Block Property is Invalid with the E2925A");
        break;
      }                         /*lint !e788 not all enums used ... switch */
    }                           /* else */
  }

  B_ERRETURN(B_TRY_RET);
}


/* set the value of a property for the next programming of an block line */
b_errtype EXPORT BestMasterBlockPropSet(
    b_handletype handle,
    b_blkproptype blk_prop,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestMasterBlockPropSet [mbprpset]");

  B_TRY_VARS_NO_PROG;

  b_int32 help;
  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);

    /* dynamic capability checking concerning values of parameters */
    B_TRY(BestParamCheck(handle, B_PARAM_MASTER_BLOCK, (b_int32)blk_prop, value));

    /* Do not allow dual address cycles as bus command, use special property
     * in E2926 only. This is currently not checked by the above
     * BestParamCheck() */
    B_TRY_FAIL(((blk_prop == B_BLK_BUSCMD) && (value == 0x0d)) ?
      B_E_MBLOCKCMD_DAC : B_E_OK);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestAbstractPropLongSet(handle, CMD_EXEFHIF_MBLK_PROP_SET,
          (b_int8) blk_prop, value));
    }
    else
    {
      switch (blk_prop)
      {
      case B_BLK_BUSADDR:
        B_TRY(BestBasicBlockWrite(handle, BUS_ADDR_LO, (b_int8ptr) & value, 4UL));
        break;

      case B_BLK_BUSCMD:
        /* DAC is checked above for E2925 and E2926 */
        B_TRY(BestBasicBlockWrite(handle, BUS_CMD, (b_int8ptr) & value, 1UL));
        break;

      case B_BLK_BYTEN:
        B_TRY(BestBasicBlockWrite(handle, BYTEN, (b_int8ptr) & value, 1UL));
        break;

      case B_BLK_INTADDR:
        help = value >> 2;
        B_TRY(BestBasicBlockWrite(handle, INT_ADDR, (b_int8ptr) & help, 4UL));
        break;

      case B_BLK_NOFDWORDS:
        B_TRY(BestBasicBlockWrite(handle, N_OF_TRANSFERS, (b_int8ptr) & value, 4UL));
        break;

      case B_BLK_ATTRPAGE:
        B_TRY(BestBasicBlockWrite(handle, MATTR_RUN_PAGE_NUM, (b_int8ptr) & value, 4UL));
        break;

      case B_BLK_COMPOFFS:
        help = value >> 2;
        B_TRY(BestBasicBlockWrite(handle, COMPARE_OFFSET, (b_int8ptr) & help, 4UL));
        break;

      case B_BLK_COMPFLAG:
        B_TRY(BestBasicBlockWrite(handle, COMPARE_CONTROL, (b_int8ptr) & value, 1UL));
        break;

      default:
        B_TRY_FCT_PARAM_R(2, ABORT, "Block Property is Invalid with the E2925A");
        break;
      }                         /*lint !e788 not all enums used */
    }
  }

  B_ERRETURN(B_TRY_RET);
}


#ifndef BEST_FIRMWARE

/* does not exist in firmware */

/* belongs to convenience functions, programs one complete line of properties */
b_errtype EXPORT BestMasterAllBlock1xProg(
    b_handletype handle,
    b_int32 busaddr,
    b_int32 buscmd,
    b_int32 byten,
    b_int32 intaddr,
    b_int32 nofdwords,
    b_int32 attrpage,
    b_int32 compflag,
    b_int32 compoffset
)
{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* Caution: E2926A has some more block-properties. We set them to default
     * (chris). */
    B_TRY(BestMasterBlockPropDefaultSet(handle));

    /* program block properties */
    B_TRY(BestMasterBlockPropSet(handle, B_BLK_BUSADDR, busaddr));
    B_TRY(BestMasterBlockPropSet(handle, B_BLK_BUSCMD, buscmd));
    B_TRY(BestMasterBlockPropSet(handle, B_BLK_INTADDR, intaddr));
    B_TRY(BestMasterBlockPropSet(handle, B_BLK_BYTEN, byten));
    B_TRY(BestMasterBlockPropSet(handle, B_BLK_NOFDWORDS, nofdwords));
    B_TRY(BestMasterBlockPropSet(handle, B_BLK_ATTRPAGE, attrpage));
    B_TRY(BestMasterBlockPropSet(handle, B_BLK_COMPOFFS, compoffset));
    B_TRY(BestMasterBlockPropSet(handle, B_BLK_COMPFLAG, compflag));

    /* program block line */
    B_TRY(BestMasterBlockProg(handle));
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


/* programms a block line with the actual value of the property registers */
b_errtype EXPORT BestMasterBlockProg(b_handletype handle)
{
  B_DECLARE_FUNCNAME ("BestMasterBlockProg [mbprog]"); 
  B_TRY_VARS_NO_PROG;
  b_int16 attrib;
  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestBasicCommand(handle, CMD_EXE_MBLKPROG, NULL, 0, NULL, NULL));
    }
    else
    {
      attrib = PCI_BLOCK_PROG;
      B_TRY(BestBasicBlockWrite(handle, PCI_COMMAND, (b_int8ptr) & attrib, 2UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* programms a block line with the actual value of the property registers */
b_errtype EXPORT BestMasterBlockLineProg(
    b_handletype handle,
    b_int32 page_num,
    b_int32 offset
)
{
  B_DECLARE_FUNCNAME ("BestMasterBlockLineProg [mblprog]"); 
  B_TRY_VARS_NO_PROG;
  b_int8 zw8[2], zw;
  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

#ifndef BEST_FIRMWARE
    
    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_BLOCK, &GenInfo));
    B_TRY (BestIs2925 (handle) ? B_E_NOT_E2925A : B_E_OK);
    
    assert(GenInfo->num_pages == 15);
    assert(GenInfo->num_blocks == 255 /* ==15*17 */ );

    /* Check range of page_num */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_MBLOCK_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of offset */
    if (offset >= GenInfo->num_blocks)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_OFFSET);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_blocks - 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
      B_TRY_ERROR(B_E_RANGE);
    }

#endif  /* BEST_FIRMWARE */
    
    zw = (b_int8) page_num;
    (void) BestByteCopy(zw8 + 0, &zw, 1UL);
    zw = (b_int8) offset;
    (void) BestByteCopy(zw8 + 1, &zw, 1UL);
    B_TRY(BestBasicCommand(handle, CMD_EXE_MBLKLINEPROG,
			   zw8, IN_EXE_MBLKLINEPROG, NULL, NULL));
  }

  B_ERRETURN(B_TRY_RET);
}


/* programms an End-of-block line in block memory */
b_errtype EXPORT BestMasterBlockEndProg(
    b_handletype handle,
    b_int32 page_num,
    b_int32 offset
)
{
  B_DECLARE_FUNCNAME ("BestMasterBlockEndProg [mbeprog]"); 
  B_TRY_VARS_NO_PROG;
  b_int8 zw8[2], zw;
  
  B_TRY_BEGIN {
    
#ifndef BEST_FIRMWARE
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_BLOCK, &GenInfo));
    B_TRY (BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);

    assert(GenInfo->num_pages == 15);
    assert(GenInfo->num_blocks == 255);

    /* Check range of page_num */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_MBLOCK_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of offset */
    if (offset >= GenInfo->num_blocks)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_OFFSET);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_blocks - 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
      B_TRY_ERROR(B_E_RANGE);
    }

#endif  /* BEST_FIRMWARE */
    
    zw = (b_int8) page_num;
    (void) BestByteCopy(zw8 + 0, &zw, 1UL);
    zw = (b_int8) offset;
    (void) BestByteCopy(zw8 + 1, &zw, 1UL);
    B_TRY(BestBasicCommand(handle, CMD_EXE_MBLKENDPROG,
			   zw8, IN_EXE_MBLKENDPROG,
			   NULL, NULL));
  }

  B_ERRETURN(B_TRY_RET);
}

b_errtype EXPORT BestMasterBlockLineRead(
    b_handletype handle,
    b_int32 page_num,
    b_int32 offset,
    b_int32 * eop
)
{
  B_DECLARE_FUNCNAME("BestMasterBlockLineRead [mblread]");

  B_TRY_VARS_NO_PROG;
  b_int8 zw8[2], zw, eop8;
  b_int16 size;

  B_TRY_BEGIN
  {

#ifndef BEST_FIRMWARE
    
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    B_TRY_FCT_PARAM_NULL_POINTER(eop);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_BLOCK, &GenInfo));
    B_TRY (BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);

    /* We only check against max-values here. Exact check is done in
     * firmware */
    assert(GenInfo->num_pages == 15);
    assert(GenInfo->num_blocks == 255);

    /* Check range of page_num */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_MBLOCK_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of offset */
    if (offset >= GenInfo->num_blocks)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_OFFSET);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_blocks - 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
      B_TRY_ERROR(B_E_RANGE);
    }

#endif  /* BEST_FIRMWARE */
    
    zw = (b_int8) page_num;
    (void) BestByteCopy(zw8 + 0, &zw, 1UL);
    zw = (b_int8) offset;
    (void) BestByteCopy(zw8 + 1, &zw, 1UL);
    size = OUT_EXE_MBLK_LINEREAD;
    B_TRY(BestBasicCommand(handle, CMD_EXE_MBLK_LINEREAD,
			   zw8, IN_EXE_MBLK_LINEREAD, &eop8, &size));
    assert(size == 1);
    (void) BestByteCopy(&zw, &eop8, 1UL);
    *eop = zw;
  }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * Exerciser Run Functions
 * -------------------------------------------------------------------------- */

/* starts the master with the actual property registers */
b_errtype EXPORT BestMasterBlockRun(
    b_handletype handle
)
{
  B_DECLARE_FUNCNAME ("BestMasterBlockRun [mbrun]"); 
  
  B_TRY_VARS_NO_PROG;
  /* start transfer */
  b_int16 zw = RUN_BLOCK;
  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestBasicCommand(handle, CMD_EXEFHIF_MBLKRUN, NULL, 0, NULL, NULL));
    }

    else
    {
      B_TRY(BestBasicBlockWrite(handle, PCI_COMMAND, (b_int8ptr) & zw, 2UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* starts the master  with a block page */
b_errtype EXPORT BestMasterBlockPageRun(
    b_handletype handle,
    b_int32 page_num)
{
  B_DECLARE_FUNCNAME ("BestMasterBlockPageRun [mbpgrun]"); 
  B_TRY_VARS_NO_PROG;
  b_int8 zw8 = (b_int8)page_num;
  b_int16 zw = RUN_BLOCK_PAGE;
  
  B_TRY_BEGIN {
#ifndef BEST_FIRMWARE
    
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_BLOCK, &GenInfo));

    /* Check range of page_num (==127 or 15) */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_MBLOCK_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }

#endif  /* BEST_FIRMWARE */
    
    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestBasicCommand(handle, CMD_EXE_MBLKPGRUN,
          &zw8, IN_EXE_MBLKPGRUN, NULL, NULL));
    }

    else
    {
      B_TRY(BestBasicBlockWrite(handle, MBLK_RUN_PAGE_NUM, &zw8, 1UL));

      /* start transfer */
      B_TRY(BestBasicBlockWrite(handle, PCI_COMMAND, (b_int8ptr) & zw, 2UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}

/* starts the master  with a block page again, no more initialization */
b_errtype EXPORT BestMasterBlockPageRerun(
    b_handletype handle,
    b_int32 page_num)
{
  B_DECLARE_FUNCNAME ("BestMasterBlockPageRerun [mbpgrerun]"); 
  B_TRY_VARS_NO_PROG;
  b_int8 zw8 = (b_int8)page_num;
  /* b_int16 zw = RUN_BLOCK_PAGE; */
  
  B_TRY_BEGIN {
#ifndef BEST_FIRMWARE
    
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_BLOCK, &GenInfo));

    /* Check range of page_num (==127 or 15) */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_MBLOCK_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }

#endif  /* BEST_FIRMWARE */
    
    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestBasicCommand(handle, CMD_EXE_MBLKPGRERUN,
          &zw8, IN_EXE_MBLKPGRUN, NULL, NULL));
    }

    else
    {
      B_TRY_FAIL (B_E_NOT_E2925A);
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* stops the master */
b_errtype EXPORT BestMasterStop(
    b_handletype handle)
{
  B_DECLARE_FUNCNAME ("BestMasterStop [mstop]"); 
  B_TRY_VARS_NO_PROG;
  b_int16 zw = STOP;
  
  B_TRY_BEGIN {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestBasicCommand(handle, CMD_EXEFHIF_MSTOP, NULL, 0, NULL, NULL));
    }
    else
    {
      B_TRY(BestBasicBlockWrite(handle, PCI_COMMAND, (b_int8ptr) & zw, 2UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * PCI Master Behaviour Programming Functions
 * ------------------------------------------------------------------------- */

/* initializes a attrigute page */
b_errtype EXPORT BestMasterAttrPageInit(
    b_handletype handle,
    b_int32 page_num
)
{
  B_DECLARE_FUNCNAME ("BestMasterAttrPageInit [mapginit]"); 
  B_TRY_VARS_NO_PROG;
  b_int8 zw8;
  
  B_TRY_BEGIN {
#ifndef BEST_FIRMWARE
    
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_ATTR, &GenInfo));

    /* Check range of page_num */
    /* For E2926A the exact check is done in firmware ! */
    assert(GenInfo->num_pages >= 7);

    /* Check range of offset page_num should be in range: E2925A: 1,..,255
     * E2926A: 1,..,7 resp. 1,..,63 depending on the AttibutePageSize */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_MATTR_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }

#endif  /* BEST_FIRMWARE */
    
    if (Best16BitRegisterFile(handle))
    {
      zw8 = (b_int8)page_num;
      B_TRY(BestBasicCommand(handle, CMD_EXE_MATTR_PAGEINIT,
          &zw8, IN_EXE_MATTR_PAGEINIT, NULL, NULL));

    }
    else
    {
      B_TRY(BestBasicBlockWrite(handle, MATTR_PROG_PAGE_NUM,
          (b_int8ptr) & page_num, 4UL));
      B_TRY(BestBasicBlockWrite(handle, MATTR_CLEAR,
          (b_int8ptr) & page_num, 1UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}


#ifndef BEST_FIRMWARE

/* sets the programming pointes in the specified page to the specified offset*/
b_errtype EXPORT BestMasterAttrPtrSet(
    b_handletype handle,
    b_int32 page_num,
    b_int32 offset
)
{
  B_TRY_VARS_NO_PROG;
  b_int8 zw8[2], zw;
  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_ATTR, &GenInfo));

    /* Range checking */
    /* Remark: For concatenated pages, the offset may be bigger than 0x1f in
     * E2926A */

    if (Best16BitRegisterFile(handle))
    {
      /* We only check against max-values here. Exact check is done in
       * firmware */
      assert(GenInfo->num_pages == 63);
      assert(GenInfo->num_blocks == 252);

      /* Check range of page_num */
      if ((page_num == 0) || (page_num > GenInfo->num_pages))
      {
        BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
        BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_MBLOCK_PAGE);
        BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
        BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
        BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
        B_TRY_ERROR(B_E_RANGE);
      }
      /* Check range of offset */
      if (offset >= GenInfo->num_blocks)
      {
        BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
        BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_OFFSET);
        BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
        BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_blocks - 1UL);
        BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
        B_TRY_ERROR(B_E_RANGE);
      }

      zw = (b_int8) page_num;
      (void) BestByteCopy(zw8 + 0, &zw, 1UL);
      zw = (b_int8) offset;
      (void) BestByteCopy(zw8 + 1, &zw, 1UL);
      B_TRY(BestBasicCommand(handle, CMD_EXE_MATTR_PTRSET,
          zw8, IN_EXE_MATTR_PTRSET, NULL, NULL));
    }

    else
    {
      assert(GenInfo->num_pages == 255);
      assert(GenInfo->blocks_per_page == 32);

      /* Check range of page_num */
      if ((page_num == 0) || (page_num > GenInfo->num_pages))
      {
        BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
        BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_MBLOCK_PAGE);
        BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
        BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
        BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
        B_TRY_ERROR(B_E_RANGE);
      }
      /* Check range of offset */
      if (offset >= GenInfo->blocks_per_page)
      {
        BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
        BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_OFFSET);
        BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
        BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->blocks_per_page - 1UL);
        BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
        B_TRY_ERROR(B_E_RANGE);
      }

      B_TRY(BestBasicBlockWrite(handle, MATTR_PROG_PAGE_NUM,
          (b_int8ptr) & page_num, 4UL));
      B_TRY(BestBasicBlockWrite(handle, MATTR_PROG_OFFSET,
          (b_int8ptr) & offset, 4UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


#ifndef BEST_FIRMWARE
/* does not exist in firmware implementation */
/* sets all attribute properties to their specifiec default value */
b_errtype EXPORT BestMasterAttrPropDefaultSet(
    b_handletype handle
)
{
  B_TRY_VARS_NO_PROG;
  b_int32 value = M_LOCK_UNLOCK_BIT;
  b_int32 i;
  b_param_infotype *ParamInfo;
  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    if (Best16BitRegisterFile(handle))
    {
      /* Get pointer to generic info */
      B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_ATTR, &GenInfo));

      for (i = 0; i < GenInfo->num_elem; i++)
      {
        /* Get pointer to i-th (existing) property */
        B_TRY(BestParamInfoGet(handle, B_PARAM_MASTER_ATTR,
	    i, &ParamInfo, (b_int32)B_INDEX_SEARCH));

        /* Set it to default */
        B_TRY(BestMasterAttrPropSet(handle,
            ParamInfo->proptyp.mattrprop,
            ParamInfo->defaultval));
      }
    }

    else
    {
      B_TRY(BestBasicBlockWrite(handle, M_ATTR, (b_int8ptr) & value, 4UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


/* sets the value of the master attribute property register */
b_errtype EXPORT BestMasterAttrPropSet(
    b_handletype handle,
    b_mattrproptype mattrprop,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME("BestMasterAttrPropSet [maprpset]");

  B_TRY_VARS_NO_PROG;
  
  B_TRY_BEGIN {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* dynamic capability checking concerning values of parameters */
    B_TRY(BestParamCheck(handle, B_PARAM_MASTER_ATTR, (b_int32)mattrprop, value));

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestAbstractPropLongSet(handle, CMD_EXE_MATTR_PROP_SET,
          (b_int8) mattrprop, value));
    }

    else {

#ifndef BEST_FIRMWARE
      
      b_int32 regval;

      /* special hack for relreq, which is handled differently in the E2925 */
      if (mattrprop == B_M_RELREQ)
      {
        b_hwtype HWType;
        B_TRY(BestGetHWFromHandle(handle, &HWType));
        if (HWType == B_HW_E2925A || HWType == B_HW_E2925A_DEEP)
        {
          if (value == B_RELREQ_ON) /* this means ON */
            value = 1;
          else if (value == B_RELREQ_OFF ||
            value == B_DRELREQ_OFF) /* these both mean OFF */
            value = 0;
        }
      }

      B_TRY(BestBasicBlockRead(handle, M_ATTR, (b_int8ptr) & regval, 4UL));

      switch (mattrprop)
      {
      case B_M_WAITS:
        regval &= ~M_WAITS_BITS;/* clear all waits */
        regval |= (value & M_WAITS_BITS);
        break;

      case B_M_DOLOOP:
        if (value == 0)
        {
          regval &= ~M_DOLOOP_BIT;
        }
        else
        {
          regval |= M_DOLOOP_BIT;
        }
        break;

      case B_M_DSERR:
        if (value == 0)
        {
          regval &= ~M_DSERR_BIT;
        }
        else
        {
          regval |= M_DSERR_BIT;
        }
        break;

      case B_M_LAST:
        if (value == 0)
        {
          regval &= ~M_LAST_BIT;
        }
        else
        {
          regval |= M_LAST_BIT;
        }
        break;

      case B_M_RELREQ:
        if (value == 0)
        {
          regval &= ~M_RELREQ_BIT;
        }
        else
        {
          regval |= M_RELREQ_BIT;
        }
        break;

      case B_M_DPERR:
        if (value == 0)
        {
          regval &= ~M_DPERR_BIT;
        }
        else
        {
          regval |= M_DPERR_BIT;
        }
        break;

      case B_M_APERR:
        if (value == 0)
        {
          regval &= ~M_APERR_BIT;
        }
        else
        {
          regval |= M_APERR_BIT;
        }
        break;

      case B_M_DWRPAR:
        if (value == 0)
        {
          regval &= ~M_DWRPAR_BIT;
        }
        else
        {
          regval |= M_DWRPAR_BIT;
        }
        break;

      case B_M_AWRPAR:
        if (value == 0)
        {
          regval &= ~M_AWRPAR_BIT;
        }
        else
        {
          regval |= M_AWRPAR_BIT;
        }
        break;

      case B_M_WAITMODE:
        if (value == B_WAITMODE_STABLE)
        {
          regval &= ~M_DTOGGLE_BIT;
        }
        else
        {
          regval |= M_DTOGGLE_BIT;
        }
        break;

      case B_M_STEPMODE:
        if (value == 0)
        {
          regval &= ~M_ATOGGLE_BIT;
        }
        else
        {
          regval |= M_ATOGGLE_BIT;
        }
        break;

      case B_M_LOCK:
        /* at first clear the lock bits all */
        regval &= ~M_LOCK_BITS;

        switch (value)
        {
        case B_LOCK_UNCHANGE:
          break;
        case B_LOCK_LOCK:
          regval |= M_LOCK_LOCK_BIT;
          break;
        case B_LOCK_HIDELOCK:
          regval |= M_LOCK_HIDELOCK_BIT;
          break;
        case B_LOCK_UNLOCK:
          regval |= M_LOCK_UNLOCK_BIT;
          break;
        default:
          B_TRY_FCT_PARAM_R(3, ABORT, "Invalid Attribute Value");
        }

        break;
      default:
        B_TRY_FCT_PARAM_R(2, ABORT, "Invalid Property");
      }                         /*lint !e788 not all enums used in switch */

      B_TRY(BestBasicBlockWrite(handle, M_ATTR, (b_int8ptr) & regval, 4UL));

#else /* BEST_FIRMWARE */
      B_TRY_ERROR (B_EFW_FUNC);

#endif  /* BEST_FIRMWARE */
      
    }                           /* else */
  }

  B_ERRETURN(B_TRY_RET);
}


/* returns the value ot a master attribute property register */
b_errtype EXPORT BestMasterAttrPropGet(
    b_handletype handle,
    b_mattrproptype mattrprop,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestMasterAttrPropGet [maprpget]");
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);
    B_TRY_FCT_PARAM_NULL_POINTER(value);

    /* Check range of mattrprop */
    B_TRY(BestParamProbe(handle, B_PARAM_MASTER_ATTR, (b_int32)mattrprop));

    if (Best16BitRegisterFile(handle))
    {
      /* Get current value */
      B_TRY(BestAbstractPropLongGet(handle,
          CMD_EXE_MATTR_PROP_GET,
          (b_int8) mattrprop,
          value));
    }

    else {

#ifndef BEST_FIRMWARE
      
      b_int32 regval;
      
      B_TRY(BestBasicBlockRead(handle, M_ATTR, (b_int8ptr) & regval, 4UL));

      switch (mattrprop)
      {
      case B_M_WAITS:
        *value = (regval & M_WAITS_BITS);
        break;

      case B_M_DOLOOP:
        *value = ((regval & M_DOLOOP_BIT) == 0) ? 0 : 1;
        break;

      case B_M_LAST:
        *value = ((regval & M_LAST_BIT) == 0) ? 0 : 1;
        break;
      case B_M_RELREQ:
        *value = ((regval & M_RELREQ_BIT) == 0) ? 0 : 1;
        break;

      case B_M_DPERR:
        *value = ((regval & M_DPERR_BIT) == 0) ? 0 : 1;
        break;
      case B_M_DSERR:
        *value = ((regval & M_DSERR_BIT) == 0) ? 0 : 1;
        break;
      case B_M_APERR:
        *value = ((regval & M_APERR_BIT) == 0) ? 0 : 1;
        break;
      case B_M_DWRPAR:
        *value = ((regval & M_DWRPAR_BIT) == 0) ? 0 : 1;
        break;
      case B_M_AWRPAR:
        *value = ((regval & M_AWRPAR_BIT) == 0) ? 0 : 1;
        break;
      case B_M_WAITMODE:
        *value = ((regval & M_DTOGGLE_BIT) == 0) ? 0 : 1;
        break;
      case B_M_STEPMODE:
        *value = ((regval & M_ATOGGLE_BIT) == 0) ? 0 : 1;
        break;
      case B_M_LOCK:
        *value = (regval & M_LOCK_BITS) >> 14;
        break;
      default:
        B_TRY_FCT_PARAM_R(2, ABORT, "Invalid Property");
      } /*lint !e788 not all enums used in switch */

      /* special hack for relreq, which is handled differently in the E2925 */
      if (mattrprop == B_M_RELREQ)
      {
        b_hwtype HWType;
        B_TRY(BestGetHWFromHandle(handle, &HWType));
        if (HWType == B_HW_E2925A || HWType == B_HW_E2925A_DEEP)
        {
          if (*value == 1)      /* this means ON */
            *value = B_RELREQ_ON;
          else if (*value == 0) /* this means OFF */
            *value = B_RELREQ_OFF;
        }
      }

#else /* BEST_FIRMWARE */

      B_TRY_ERROR (B_EFW_FUNC);

#endif  /* BEST_FIRMWARE */
      
    }
  }

  B_ERRETURN(B_TRY_RET);
}


#ifndef BEST_FIRMWARE

/* belongs to convenience functions, programs a complete attribute line */
b_errtype EXPORT BestMasterAllAttr1xProg(
    b_handletype handle,
    b_int32 doloop,
    b_int32 waits,
    b_int32 last,
    b_int32 dperr,
    b_int32 dserr,
    b_int32 aperr,
    b_int32 dwrpar,
    b_int32 awrpar,
    b_int32 relreq,
    b_int32 stepmode,
    b_int32 waitmode,
    b_int32 lock
)
{
  B_TRY_VARS_NO_PROG;
  b_errtype err;
  B_TRY_BEGIN
  {
    /* license checking */
    B_LICENSECHECK(B_CAPABILITY_EXERCISER);

    /* E2926A has more properties, we set them to default */
    B_TRY(BestMasterAttrPropDefaultSet(handle));

    B_TRY(BestMasterAttrPropSet(handle, B_M_DOLOOP, doloop));
    B_TRY(BestMasterAttrPropSet(handle, B_M_LAST, last));
    B_TRY(BestMasterAttrPropSet(handle, B_M_WAITS, waits));
    B_TRY(BestMasterAttrPropSet(handle, B_M_DPERR, dperr));
    B_TRY(BestMasterAttrPropSet(handle, B_M_DSERR, dserr));
    B_TRY(BestMasterAttrPropSet(handle, B_M_APERR, aperr));
    B_TRY(BestMasterAttrPropSet(handle, B_M_DWRPAR, dwrpar));
    B_TRY(BestMasterAttrPropSet(handle, B_M_AWRPAR, awrpar));
    B_TRY(BestMasterAttrPropSet(handle, B_M_RELREQ, relreq));
    B_TRY(BestMasterAttrPropSet(handle, B_M_STEPMODE, stepmode));
    B_TRY(BestMasterAttrPropSet(handle, B_M_WAITMODE, waitmode));
    B_TRY(BestMasterAttrPropSet(handle, B_M_LOCK, lock));

    /* program attribute phase */
    B_TRY(BestMasterAttrPhaseProg(handle));
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


#ifndef BEST_FIRMWARE

/* reads the attribute properties from the actual position */
b_errtype EXPORT BestMasterAttrPhaseRead(
    b_handletype handle
)
{
  B_TRY_VARS_NO_PROG;
  b_int16 attrib;
  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestBasicCommand(handle, CMD_EXE_MATTR_PHASEREAD,
          NULL, 0, NULL, NULL));
    }
    else
    {
      attrib = MASTER_PHASE_READ;
      B_TRY(BestBasicBlockWrite(handle, PCI_COMMAND, (b_int8ptr) & attrib, 2UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


#ifndef BEST_FIRMWARE

/* programms a phase in the attribute memory */
b_errtype EXPORT BestMasterAttrPhaseProg(
    b_handletype handle
)
{
  B_TRY_VARS_NO_PROG;
  b_int16 attrib;
  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestBasicCommand(handle, CMD_EXE_MATTR_PHASEPROG,
          NULL, 0, NULL, NULL));
    }

    else
    {
      attrib = MASTER_PHASE_PROG;
      B_TRY(BestBasicBlockWrite(handle, PCI_COMMAND, (b_int8ptr) & attrib, 2UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


b_errtype EXPORT BestMasterAttrLineProg(
    b_handletype handle,
    b_int32 page_num,
    b_int32 offset
)
{
  B_DECLARE_FUNCNAME ("BestMasterAttrLineProg [malprog]"); 
  B_TRY_VARS_NO_PROG;
  b_int8 page8, offset8, zw8 [2];

  B_TRY_BEGIN {

#ifndef BEST_FIRMWARE
    
    B_TRY (BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_ATTR, &GenInfo));

    /* Check range of page_num, exact check is done in FW */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_MATTR_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of offset */
    if (offset >= GenInfo->num_blocks)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_OFFSET);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_blocks - 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
      B_TRY_ERROR(B_E_RANGE);
    }

#endif  /* BEST_FIRMWARE */
    
    page8 = (b_int8) page_num;
    offset8 = (b_int8) offset;
    
    (void) BestByteCopy(zw8 + 0, &page8, 1UL);
    (void) BestByteCopy(zw8 + 1, &offset8, 1UL);

    B_TRY(BestBasicCommand(handle, CMD_EXE_MATTR_LINEPROG,
			   zw8, IN_EXE_MATTR_LINEPROG, NULL, NULL));
  }

  B_ERRETURN(B_TRY_RET);
}


#ifndef E2921A_MIN

/* programming properties of one group only */
b_errtype EXPORT BestMasterAttrGroupLineProg(
    b_handletype handle,
    b_mattrgrouptype group,
    b_int32 page_num,
    b_int32 offset
)
{
  B_DECLARE_FUNCNAME ("BestMasterAttrGroupLineProg [maglprog]");
  B_TRY_VARS_NO_PROG;
  b_int8 group8, page8, offset8, zw8 [3];

  B_TRY_BEGIN {

#ifndef BEST_FIRMWARE
    
    B_TRY (BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_ATTR, &GenInfo));

    /* Check range of page_num, exact check is done in FW */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_MATTR_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of offset */
    if (offset >= GenInfo->num_blocks)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_OFFSET);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_blocks - 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of group */
    if (group >= B_MATTR_GRP_NUM)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_GROUP);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, B_MATTR_GRP_NUM - 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_5, (b_int32)group);
      B_TRY_ERROR(B_E_RANGE);
    }

#endif  /* BEST_FIRMWARE */
    
    group8 = (b_int8) group;
    page8 = (b_int8) page_num;
    offset8 = (b_int8) offset;

    (void) BestByteCopy(zw8 + 0, &group8, 1UL);
    (void) BestByteCopy(zw8 + 1, &page8, 1UL);
    (void) BestByteCopy(zw8 + 2, &offset8, 1UL);

    B_TRY(BestBasicCommand(handle, CMD_EXE_MATTR_GRPLINEPROG,
			   zw8, IN_EXE_MATTR_GRPLINEPROG, NULL, NULL));
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* E2921A_MIN */


b_errtype EXPORT BestMasterAttrLineRead(
    b_handletype handle,
    b_int32 page_num,
    b_int32 offset
)
{
  B_DECLARE_FUNCNAME ("BestMasterAttrLineRead [malread]"); 
  B_TRY_VARS_NO_PROG;
  b_int8 page8, offset8, zw8 [2];

  B_TRY_BEGIN {

#ifndef BEST_FIRMWARE
    
    B_TRY (BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_ATTR, &GenInfo));

    /* Check range of page_num, exact check is done in FW */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_MATTR_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of offset */
    if (offset >= GenInfo->num_blocks)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_OFFSET);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_blocks - 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
      B_TRY_ERROR(B_E_RANGE);
    }

#endif  /* BEST_FIRMWARE */
    
    page8 = (b_int8) page_num;
    offset8 = (b_int8) offset;

    (void) BestByteCopy(zw8 + 0, &page8, 1UL);
    (void) BestByteCopy(zw8 + 1, &offset8, 1UL);
    
    B_TRY(BestBasicCommand(handle, CMD_EXE_MATTR_LINEREAD,
			   zw8, IN_EXE_MATTR_LINEREAD, NULL, NULL));
  }

  B_ERRETURN(B_TRY_RET);
}


#ifndef E2921A_MIN

/* reading properties of one group only */
b_errtype EXPORT BestMasterAttrGroupLineRead(
    b_handletype handle,
    b_mattrgrouptype group,
    b_int32 page_num,
    b_int32 offset
)
{
  B_DECLARE_FUNCNAME ("BestMasterAttrGroupLineRead [maglread]");
  B_TRY_VARS_NO_PROG;
  b_int8 group8, page8, offset8, zw8 [3];

  B_TRY_BEGIN {

#ifndef BEST_FIRMWARE
    B_TRY (BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_ATTR, &GenInfo));

    /* Check range of page_num, exact check is done in FW */
    if ((page_num == 0) || (page_num > GenInfo->num_pages))
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_MATTR_PAGE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_pages);
      BestLastErrorParamSet(handle, B_ERRPAR_5, page_num);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of offset */
    if (offset >= GenInfo->num_blocks)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_OFFSET);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, GenInfo->num_blocks - 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
      B_TRY_ERROR(B_E_RANGE);
    }
    /* Check range of group */
    if (group >= B_MATTR_GRP_NUM)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_GROUP);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, B_MATTR_GRP_NUM - 1UL);
      BestLastErrorParamSet(handle, B_ERRPAR_5, (b_int32)group);
      B_TRY_ERROR(B_E_RANGE);
    }

#endif  /* BEST_FIRMWARE */
    
    group8 = (b_int8) group;
    page8 = (b_int8) page_num;
    offset8 = (b_int8) offset;

    (void) BestByteCopy(zw8 + 0, &group8, 1UL);
    (void) BestByteCopy(zw8 + 1, &page8, 1UL);
    (void) BestByteCopy(zw8 + 2, &offset8, 1UL);

    B_TRY(BestBasicCommand(handle, CMD_EXE_MATTR_GRPLINEREAD,
			   zw8, IN_EXE_MATTR_GRPLINEREAD, NULL, NULL));
  }

  B_ERRETURN(B_TRY_RET);
}


/* --------------------------------------------------------------------------
 * PCI Master Generic Property Programming Functions
 * -------------------------------------------------------------------------- */

/* set a generic property to value */
b_errtype EXPORT BestMasterGenPropSet(
    b_handletype handle,
    b_mastergenproptype mastergenprop,
    b_int32 value)
{
  B_DECLARE_FUNCNAME("BestMasterGenPropSet [mgprpset]");

  B_TRY_VARS_NO_PROG;
  b_int32 regnr;
  b_int8 zw;
  b_int16 wzw;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);

    /* dynamic capability checking concerning values of parameters */
    B_TRY(BestParamCheck(handle, B_PARAM_MASTER_GEN, (b_int32)mastergenprop, value));

    if (Best16BitRegisterFile(handle))
    {
      /* E2926 */
      B_TRY(BestAbstractPropLongSet(handle, CMD_EXEFHIF_MGEN_PROP_SET,
          (b_int8) mastergenprop, value));
    }                           /* if (new protocol) */
    else
    {
      /* E2925 */
      switch (mastergenprop)
      {
      case B_MGEN_RUNMODE:
        regnr = MGEN_RUNMODE;
        zw = (b_int8) value;
        B_TRY(BestBasicBlockWrite(handle, regnr, &zw, 1UL));
        break;

      case B_MGEN_REPEATMODE:
        regnr = MGEN_REPEAT;
        wzw = (b_int16) value;
        B_TRY(BestBasicBlockWrite(handle, regnr, (b_int8ptr) & wzw, 2UL));
        break;

      case B_MGEN_DELAYCTR:
        regnr = MGEN_DELAYCTR;
        wzw = (b_int16) value;
        B_TRY(BestBasicBlockWrite(handle, regnr, (b_int8ptr) & wzw, 2UL));
        break;

      case B_MGEN_LATMODE:
        regnr = MGEN_LATMODE;
        zw = (b_int8) value;
        B_TRY(BestBasicBlockWrite(handle, regnr, &zw, 1UL));
        break;

      case B_MGEN_LATCTR:
        regnr = MGEN_LATCTR;
        zw = (b_int8) value;
        B_TRY(BestBasicBlockWrite(handle, regnr, &zw, 1UL));
        break;

      case B_MGEN_MWIENABLE:
        regnr = MGEN_MWI;
        zw = (b_int8) value;
        B_TRY(BestBasicBlockWrite(handle, regnr, &zw, 1UL));
        break;

      case B_MGEN_MASTERENABLE:
        regnr = MGEN_MENABLE;
        zw = (b_int8) value;
        B_TRY(BestBasicBlockWrite(handle, regnr, &zw, 1UL));
        break;

      case B_MGEN_ATTRMODE:
        regnr = MGEN_ATTRMODE;
        zw = (b_int8) value;
        B_TRY(BestBasicBlockWrite(handle, regnr, &zw, 1UL));
        break;

      default:
        B_FCT_PARAM_CHK_R(2, ABORT, "Invalid property");
        break;
      } /*lint !e788 not all enums used in switch */
    }
  }

  B_ERRETURN(B_TRY_RET);
}


#ifndef BEST_FIRMWARE

/* sets all generic properies to their specified defautl value */
b_errtype EXPORT BestMasterGenPropDefaultSet(
    b_handletype handle
)
{
  B_TRY_VARS_NO_PROG;
  b_int32 i, conf_restore;
  b_param_infotype *ParamInfo;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);

    /* Get pointer to generic info */
    B_TRY(BestGenInfoGet(handle, B_PARAM_MASTER_GEN, &GenInfo));
    B_TRY(BestPowerUpPropGet(handle, B_PU_CONFRESTORE, &conf_restore));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      B_TRY(BestParamInfoGet(handle, B_PARAM_MASTER_GEN,
	  i, &ParamInfo, (b_int32)B_INDEX_SEARCH));

      switch (ParamInfo->proptyp.mgenprop)
      {
      case B_MGEN_LATCTR:
      case B_MGEN_MASTERENABLE:
      case B_MGEN_MWIENABLE:
      case B_MGEN_CACHELINESIZE:
      case B_MGEN_TRYBACKENABLE:
        if (conf_restore == 0)
          break;

      default:  /*lint !e616 deliberate flow into case/default */
        /* Set it to default */
        B_TRY(BestMasterGenPropSet(handle,
            ParamInfo->proptyp.mgenprop,
            ParamInfo->defaultval));
      } /*lint !e788 not all enums used in switch */
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */


/* return the value of a generic property */
b_errtype EXPORT BestMasterGenPropGet(
    b_handletype handle,
    b_mastergenproptype mastergenprop,
    b_int32 * value)
{
  B_DECLARE_FUNCNAME("BestMasterGenPropGet [mgprpget]");

  B_TRY_VARS_NO_PROG;
  b_int32 regnr;
  b_int8 zw;
  b_int16 wzw;
  b_int32 CfgDW;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_HOSTINT);
    B_TRY_FCT_PARAM_NULL_POINTER(value);

    if (Best16BitRegisterFile(handle))
    {
      /* E2926 */
      switch (mastergenprop)
      {
      case B_MGEN_MASTERENABLE:
        /* This prop belong to the config-space and thus is not held in FW ! */
        /* read bit 2 in config command register */
	B_TRY(BestConfRegGet(handle, 0x4UL, &CfgDW));
	*value = (CfgDW & 0x4UL) >> 2;
        break;

      case B_MGEN_MWIENABLE:
        /* This prop belong to the config-space and thus is not held in FW ! */
        /* read bit 4 config command register */
	B_TRY(BestConfRegGet(handle, 0x4UL, &CfgDW));
	*value = (CfgDW & 0x10UL) >> 4;
        break;

      case B_MGEN_TRYBACKENABLE:
        /* This prop belong to the config-space and thus is not held in FW ! */
        /* read bit 9 config command register */
	B_TRY(BestConfRegGet(handle, 0x4UL, &CfgDW));
	*value = (CfgDW & 0x200UL) >> 9;
        break;

      case B_MGEN_LATCTR:
        /* This prop belong to the config-space and thus is not held in FW ! */
        /* read bits 15-8 in config register 0Ch, Latency Timer */
	B_TRY(BestConfRegGet(handle, 0x0cUL, &CfgDW));
	*value = (CfgDW & 0xFF00UL) >> 8; /* Extract MSB */
        break;

      case B_MGEN_CACHELINESIZE:
        /* This prop belong to the config-space and thus is not held in FW ! */
        /* read bits 7-0 in config register 0Ch, Cache Line Size */
	B_TRY(BestConfRegGet(handle, 0x0cUL, &CfgDW));
	*value = (CfgDW & 0xFFUL);  /* Extract LSB */
        break;

      default:
        /* Get pointer to i-th (existing) property */
        B_TRY(BestParamProbe(handle, B_PARAM_MASTER_GEN, (b_int32)mastergenprop));
        /* Get current value */
        B_TRY(BestAbstractPropLongGet(handle,
				      CMD_EXEFHIF_MGEN_PROP_GET,
				      (b_int8)mastergenprop, value));
      }                         /*lint !e788 not all enums used in switch */
    }                           /* new protocol */

    else
    {
      /* E2925 */
      switch (mastergenprop)
      {
      case B_MGEN_RUNMODE:
        regnr = MGEN_RUNMODE;
        B_TRY(BestBasicBlockRead(handle, regnr, &zw, 1UL));
        *value = (b_int32) zw;
        break;
      case B_MGEN_REPEATMODE:
        regnr = MGEN_REPEAT;
        B_TRY(BestBasicBlockRead(handle, regnr, (b_int8ptr) & wzw, 2UL));
        *value = (b_int32) wzw;
        break;
      case B_MGEN_DELAYCTR:
        regnr = MGEN_DELAYCTR;
	B_TRY(BestBasicBlockRead(handle, regnr, (b_int8ptr) & wzw, 2UL));
        *value = (b_int32) wzw;
        break;
      case B_MGEN_LATMODE:
        regnr = MGEN_LATMODE;
        B_TRY(BestBasicBlockRead(handle, regnr, &zw, 1UL));
        *value = (b_int32) zw;
        break;
      case B_MGEN_LATCTR:
        regnr = MGEN_LATCTR;
        B_TRY(BestBasicBlockRead(handle, regnr, &zw, 1UL));
        *value = (b_int32) zw;
        break;
      case B_MGEN_MWIENABLE:
        regnr = MGEN_MWI;
        B_TRY(BestBasicBlockRead(handle, regnr, &zw, 1UL));
        *value = (b_int32) zw;
        break;
      case B_MGEN_MASTERENABLE:
        regnr = MGEN_MENABLE;
        B_TRY(BestBasicBlockRead(handle, regnr, &zw, 1UL));
        *value = (b_int32) zw;
        break;
      case B_MGEN_ATTRMODE:
        regnr = MGEN_ATTRMODE;
        B_TRY(BestBasicBlockRead(handle, regnr, &zw, 1UL));
        *value = (b_int32) zw;
        break;
      default:
        B_FCT_PARAM_CHK_R(2, ABORT, "Invalid property");
      } /*lint !e788 not all enums used in switch */
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#ifndef BEST_FIRMWARE 

b_errtype EXPORT BestMasterCondStartPattSet(
    b_handletype handle,
    b_charptrtype pattern
)
{
  B_DECLARE_FUNCNAME ("BestMasterCondStartPattSet [mcspset]");
  B_TRY_VARS_NO_PROG;
  char *ch = "";
  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestPattSet(handle, (b_int32)B_PATT_FAST_0, pattern));
    }
    else
    {
      B_TRY(BestBasicBlockWrite(handle, INT_PATTERN, (b_int8ptr) ch, 1UL));
      B_TRY(BestBasicBlockWrite(handle, INT_PATTERN, (b_int8ptr) pattern,
          strlen(pattern) + 1UL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}

#endif  /* BEST_FIRMWARE */

#endif  /* E2921A_MIN */


#if 0                           /* BestMasterBlockPrepReStore() not yet used */
/* helper function for saving and restoring the Block-preparation register */
static b_errtype BestMasterBlockPrepReStore(b_handletype handle, b_int8 Store)
{
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* Store=1 means store, Store=0 means restore preparation register */
    if(BestIs2925(handle))
    {
      B_TRY_ERROR(B_E_NOT_E2925A);
    }
    else
    {
      b_int8 zw8;
      (void) BestByteCopy(&zw8, &Store, 1UL);
      B_TRY(BestBasicCommand(handle, CMD_EXE_MBLKPREPRESTORE, &zw8, IN_EXE_MBLKPREPRESTORE, NULL, NULL));
    }
  }

  B_ERRETURN(B_TRY_RET);
}
#endif          /* BestMasterBlockPrepReStore() not yet used */


/* --------------------------------------------------------------------------
 * Accessing Master Byteenable Memory
 * -------------------------------------------------------------------------- */

/* programming into memory */
b_errtype EXPORT BestMasterByteEnableProg(
    b_handletype handle,
    b_int32 offset,
    b_int32 value
)
{
  B_DECLARE_FUNCNAME ("BestMasterByteEnableProg [mbytenprog]");
  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    
    B_TRY (BestIs2925(handle) ? B_E_NOT_E2925A : B_E_OK);
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    /* this function exists in the E2926 only */
    /* Check range for offset  0,..15 are reserved for compatibility
     * reasons */
    if ((offset < 16) || (offset > 255)) {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_OFFSET);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 16UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, 255UL);
      BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
      B_TRY_ERROR (B_E_RANGE);
    }
      
    /* Check range of value */
    if (value > 255) {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_VALUE);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, 255UL);
      BestLastErrorParamSet(handle, B_ERRPAR_5, value);
      B_TRY_ERROR (B_E_RANGE);
    }

    B_TRY(BestAbstractPropByteSet(handle, CMD_EXE_MBYTEENABLE_PROG,
				  (b_int8) offset, value));
  }

  B_ERRETURN(B_TRY_RET);
}


/* reading from memory */
b_errtype EXPORT BestMasterByteEnableRead(
    b_handletype handle,
    b_int32 offset,
    b_int32 * value
)
{
  B_DECLARE_FUNCNAME("BestMasterByteEnableRead [mbytenread]");

  B_TRY_VARS_NO_PROG;

  B_TRY_BEGIN
  {
    /* license checking */
    B_TRY_LICENSE(B_CAPABILITY_EXERCISER);

    B_TRY_FCT_PARAM_NULL_POINTER(value);

    /* Check range of offset */
    if (offset > 255)
    {
      BestLastErrorParamSet(handle, B_ERRPAR_1, (b_int32)B_PARAM_CUSTOM);
      BestLastErrorParamSet(handle, B_ERRPAR_2, (b_int32)B_PCST_OFFSET);
      BestLastErrorParamSet(handle, B_ERRPAR_3, 0UL);
      BestLastErrorParamSet(handle, B_ERRPAR_4, 255UL);
      BestLastErrorParamSet(handle, B_ERRPAR_5, offset);
      B_TRY_ERROR(B_E_RANGE);
    }

    if (Best16BitRegisterFile(handle))
    {
      B_TRY(BestAbstractPropByteGet(handle, CMD_EXE_MBYTEENABLE_READ,
          (b_int8) offset, value));
    }

    else
    {
      B_TRY_ERROR(B_E_NOT_E2925A);
    }
  }

  B_ERRETURN(B_TRY_RET);
}
